<?php
require_once 'config.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

$user = getUser();
$message = '';
$db = getDB();

// Get available investment plans
$stmt = $db->query('SELECT * FROM investment_plans WHERE status = "active" ORDER BY name');
$plans = $stmt->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $plan_id = (int)$_POST['plan_id'];
    $amount = (float)$_POST['amount'];

    // Get plan details
    $stmt = $db->prepare('SELECT * FROM investment_plans WHERE id = ? AND status = "active"');
    $stmt->execute([$plan_id]);
    $plan = $stmt->fetch();

    if (!$plan) {
        $message = 'Invalid investment plan selected.';
    } elseif ($amount < $plan['min_amount'] || $amount > $plan['max_amount']) {
        $message = 'Amount must be between $' . number_format($plan['min_amount'], 2) . ' and $' . number_format($plan['max_amount'], 2);
    } elseif ($amount > $user['balance']) {
        $message = 'Insufficient balance. Your current balance is $' . number_format($user['balance'], 2);
    } else {
        // Calculate end date
        $end_date = date('Y-m-d H:i:s', strtotime('+' . $plan['duration'] . ' hours'));

        // Insert investment
        $stmt = $db->prepare('INSERT INTO investments (user_id, plan_id, amount, end_date) VALUES (?, ?, ?, ?)');
        if ($stmt->execute([$user['id'], $plan_id, $amount, $end_date])) {
            // Deduct from balance
            $stmt = $db->prepare('UPDATE users SET balance = balance - ? WHERE id = ?');
            $stmt->execute([$amount, $user['id']]);

            // Insert into investment history
            $stmt = $db->prepare('INSERT INTO investment_history (user_id, plan_id, amount) VALUES (?, ?, ?)');
            $stmt->execute([$user['id'], $plan_id, $amount]);

            $message = 'Investment created successfully! Your balance has been updated.';
        } else {
            $message = 'Error creating investment.';
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invest Now - <?php echo SITE_NAME; ?></title>
    <link rel="icon" href="user/assets/img/brand/favicon.png" type="image/x-icon">
    <link href="user/assets/css/icons.css" rel="stylesheet">
    <link href="user/assets/plugins/bootstrap/css/bootstrap.min.css" rel="stylesheet">
    <link href="user/assets/css/style.css" rel="stylesheet">
    <link href="user/assets/css/animate.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <div class="row">
            <div class="col-md-12">
                <h2>Available Investment Plans</h2>
                <p>Your current balance: <strong>$<?php echo number_format($user['balance'], 2); ?></strong></p>
                <?php if ($message): ?>
                    <div class="alert alert-info"><?php echo $message; ?></div>
                <?php endif; ?>
            </div>
        </div>

        <div class="row">
            <?php foreach ($plans as $plan): ?>
            <div class="col-md-4 mb-4">
                <div class="card">
                    <div class="card-header">
                        <h5><?php echo htmlspecialchars($plan['name']); ?></h5>
                    </div>
                    <div class="card-body">
                        <p><strong>Min Amount:</strong> $<?php echo number_format($plan['min_amount'], 2); ?></p>
                        <p><strong>Max Amount:</strong> $<?php echo number_format($plan['max_amount'], 2); ?></p>
                        <p><strong>ROI:</strong> <?php echo $plan['roi']; ?>%</p>
                        <p><strong>Duration:</strong> <?php echo $plan['duration']; ?> hours</p>

                        <form method="post">
                            <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">
                            <div class="form-group">
                                <label>Investment Amount ($)</label>
                                <input type="number" step="0.01" name="amount" class="form-control" required 
                                       min="<?php echo $plan['min_amount']; ?>" max="<?php echo $plan['max_amount']; ?>">
                            </div>
                            <button type="submit" class="btn btn-primary btn-block">Invest Now</button>
                        </form>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <script src="user/assets/plugins/jquery/jquery.min.js"></script>
    <script src="user/assets/plugins/bootstrap/js/bootstrap.min.js"></script>
</body>
</html>